
/*--------------------------------------------------------------------------*/
/*
   a_star.cpp
   Creator:			Eric Bengfort
   Date Created:	10.01.2007
   Description:		Implementation file for A* Path Finding.  Version 5.23	*/
/*--------------------------------------------------------------------------*/

#include "a_star.h"  // for class information.

SmartIntList::SmartIntList()
{
	// Does nothing!
}

bool SmartIntList::Find(int value)
{
	// Note to self:  m_Map.find(value) returns type "std::map<int,bool>::iterator"
	return (m_Map.find(value) != m_Map.end());

}
void SmartIntList::Remove(int value)
{
	std::map<int,bool>::iterator it = m_Map.find(value);
	if (it != m_Map.end())
	{
		m_Map.erase( it );
	}
}

void SmartIntList::Add(int value)
{
	m_Map[value] = true;
}

void SmartIntList::Clear()
{
	m_Map.clear();
}

int SmartIntList::Size()
{
	return (int)m_Map.size();
}

std::map<int,bool> *SmartIntList::GetPtrToMapItself()
{
	// use with caution
	return &m_Map;
}

Node::Node()
{
	m_type = EMPTY;
	m_F = 0.f;
	m_G = 0.f;
	m_H = 0.f;
	m_Parent = -1;
}

Astar::Astar(int w, int h, std::vector<Node> *externalNodeVecPtr)
{
	assert(w > 0);
	assert(h > 0);
	m_width = w;
	m_height = h;

	m_CurSolvVecGoalIndex = -1;
	m_Start = -1;
	m_End = -1;

	m_externalNodeVecPtr = externalNodeVecPtr;
	if (m_externalNodeVecPtr != NULL)
	{
		// Do not allocate a nodeVec into the class itself.
		// Let's make sure that the w and h provided are probably correct (match corresponding NodeVec Ptr)
		assert( w * h ==  (int)(*externalNodeVecPtr).size() );
		return;
	}

	const int nodes = w * h;
	m_NodeVec.reserve(nodes); // This allocates space for constant sized vector up front.
	for (int i = 0; i < nodes; ++i)
	{
		m_NodeVec.push_back( Node() ); // make a blank map. [are pushing back to internal so this is okay) [the ONLY okay place to use TrueNodeVec DIRECTLY]
	}
	assert((int)m_NodeVec.size() == nodes);
}

Astar::Astar(std::string file, char empty, char start, char end, char wall)
{
	// It wouldn't make sense to attempt to point to an external NodeVec.
	// We are reading IN from a file, so we're obviously MAKING a nodeVec!!
	// We still better set the ext to Null though since there isn't one.
	m_externalNodeVecPtr = NULL;  // It is internal.
	LoadMapFromFile(file, empty, start, end, wall);
	m_CurSolvVecGoalIndex = -1;
	m_Start = -1;
	m_End = -1;
}


int Astar::GetWidth()
{
	return m_width;
}

int Astar::GetHeight()
{
	return m_height;
}

bool Astar::IsIndexValid(int i)
{
	return (i >= 0 && i < m_width * m_height);
}

void Astar::SetWall(int i)
{
	assert( IsIndexValid(i) );
	TrueNodeVec[i].m_type = Node::WALL;
}

void Astar::SetNonWall(int i)
{
	assert( IsIndexValid(i) );
	TrueNodeVec[i].m_type = Node::EMPTY;
}

void Astar::SwapWall(int i)
{
	assert( IsIndexValid(i) );
	if (TrueNodeVec[i].m_type == Node::WALL)
	{
		TrueNodeVec[i].m_type = Node::EMPTY;
	}
	else
	{
		TrueNodeVec[i].m_type = Node::WALL;
	}
}

void Astar::SetStart(int i)
{
	assert( IsIndexValid(i) );
	m_Start = i;
}

void Astar::SetEnd(int i)
{
	assert( IsIndexValid(i) );
	m_End = i;
}

bool Astar::IsMapValid()
{
	// Has exactly one start and end.
	return ( m_Start != -1 && m_End != -1 );
}




void Astar::DebugSendMapToFile(std::string file, char empty, char start, char end, char wall, bool printSol, std::vector<int> &solVec)
{
	std::ofstream fout( file.c_str() );
	const int nodes = m_width * m_height;
	for (int i = 0; i < nodes; ++i)
	{
		if (printSol)
		{
			bool found = false;
			for (int z = 0; z < (int)solVec.size(); ++z)
			{
				if (solVec[z] == i)
				{
					fout << '@';
					found = true;
					break;
				}
			}
			if (found)
			{
				if ( (i + 1) % m_width == 0 )
				{
					fout << std::endl;
				}
				continue;
			}
		}

		// Note the limitation of this.  When saving out in this way, we cannot have both a start
		// and an end on the same square.  We really don't want to save off the start and ends anyway...
		// Update: We will output if a position is a start or an end position ONLY
		// if the square is not a wall [and therefore also empty]
		
		// Update2:  Walls will be drawn [this takes precedence over drawing if
		// a square is a start or an end].  Start/End squares are only drawn iff
		// said square is also not a wall.


		if (TrueNodeVec[i].m_type == Node::WALL)
		{
			fout << wall;
		}
		else if (m_Start == i )
		{
			fout << start;
			assert(TrueNodeVec[i].m_type == Node::EMPTY );  // Position is empty, merely displaying more info.
		}
		else if (m_End == i)
		{
			fout << end;
			assert(TrueNodeVec[i].m_type == Node::EMPTY ); // Position is empty, merely displaying more info.
		}
		else if (TrueNodeVec[i].m_type == Node::EMPTY)
		{
			// Yes, I am aware this final else if is redundant, but, it is nice
			// for a sanity/assertion check.
			fout << empty;
		}

		if ( (i + 1) % m_width == 0 )
		{
			fout << std::endl;
		}
	}

	fout.close(); // <--- Thank you!
}




bool Astar::DoesExistUp(int i)
{
	assert( IsIndexValid(i) );
	return (i >= m_width);
}

bool Astar::DoesExistDown(int i)
{
	assert( IsIndexValid(i) );
	return (i < (m_width * m_height) - m_width);
}

bool Astar::DoesExistLeft(int i)
{
	assert( IsIndexValid(i) );
	return (i % m_width != 0);
}

bool Astar::DoesExistRight(int i)
{
	assert( IsIndexValid(i) );
	return ((i + 1) % m_width != 0);
}

int Astar::OffsetUp(int i)
{
	assert( IsIndexValid(i) );
	int ret = i - m_width;
	assert( IsIndexValid(ret) );
	return ret;
}

int Astar::OffsetDown(int i)
{
	assert( IsIndexValid(i) );
	int ret = i + m_width;
	assert( IsIndexValid(ret) );
	return ret;
}

int Astar::OffsetLeft(int i)
{
	assert( IsIndexValid(i) );
	int ret = i - 1;
	assert( IsIndexValid(ret) );
	return ret;
}

int Astar::OffsetRight(int i)
{
	assert( IsIndexValid(i) );
	int ret = i + 1;
	assert( IsIndexValid(ret) );
	return ret;
}

void Astar::FindNeighbors(int i, bool useEightWay, bool playerHasSize, std::vector<int> &IntVecRet)
{
	assert( IsIndexValid(i) );
	IntVecRet.clear();

	if (DoesExistUp(i))
		IntVecRet.push_back( OffsetUp(i) );

	if (DoesExistDown(i))
		IntVecRet.push_back( OffsetDown(i) );

	if (DoesExistLeft(i))
		IntVecRet.push_back( OffsetLeft(i) );

	if (DoesExistRight(i))
		IntVecRet.push_back( OffsetRight(i) );

	if (!useEightWay)
		return;

	if (!playerHasSize)
	{
		if (DoesExistUp(i) && DoesExistLeft(i))
			IntVecRet.push_back( OffsetLeft( OffsetUp(i) ) );

		if (DoesExistUp(i) && DoesExistRight(i))
			IntVecRet.push_back( OffsetRight( OffsetUp(i) ) );

		if (DoesExistDown(i) && DoesExistLeft(i))
			IntVecRet.push_back( OffsetLeft( OffsetDown(i) ) );

		if (DoesExistDown(i) && DoesExistRight(i))
			IntVecRet.push_back( OffsetRight( OffsetDown(i) ) );
	}
	else
	{
		if (DoesExistUp(i) && DoesExistLeft(i) && TrueNodeVec[ OffsetUp(i) ].m_type != Node::WALL && TrueNodeVec[ OffsetLeft(i) ].m_type != Node::WALL )
			IntVecRet.push_back( OffsetLeft( OffsetUp(i) ) );

		if (DoesExistUp(i) && DoesExistRight(i) && TrueNodeVec[ OffsetUp(i) ].m_type != Node::WALL && TrueNodeVec[ OffsetRight(i) ].m_type != Node::WALL )
			IntVecRet.push_back( OffsetRight( OffsetUp(i) ) );

		if (DoesExistDown(i) && DoesExistLeft(i) && TrueNodeVec[ OffsetDown(i) ].m_type != Node::WALL && TrueNodeVec[ OffsetLeft(i) ].m_type != Node::WALL )
			IntVecRet.push_back( OffsetLeft( OffsetDown(i) ) );

		if (DoesExistDown(i) && DoesExistRight(i) && TrueNodeVec[ OffsetDown(i) ].m_type != Node::WALL && TrueNodeVec[ OffsetRight(i) ].m_type != Node::WALL )
			IntVecRet.push_back( OffsetRight( OffsetDown(i) ) );
	}
}


bool Astar::LetsGo(std::vector<int> &solVec, bool useEightWay, bool ignoreWalls)
{
	solVec.clear();
	bool SolutionFound = false;
	// Good Guide:
	// http://www.policyalmanac.org/games/aStarTutorial.htm
	// http://wiki.gamegardens.com/Path_Finding_Tutorial
	// http://www-cs-students.stanford.edu/%7Eamitp/gameprog.html#Paths
	assert( IsMapValid() );
	m_OpenList.Clear();
	m_ClosedList.Clear();

	// Add starting square to the open list.
	// Swapped to get around reversal below! [this is an optimization used below]
	const int StartIndex = GetEndIndex();
	const int EndIndex = GetStartIndex();

	//------------------------------------------------
	if (StartIndex == EndIndex)
	{
		solVec.push_back( EndIndex );
		return true;
	}

	if (! ignoreWalls )
	{

		if ( TrueNodeVec[ StartIndex ].m_type == Node::WALL || TrueNodeVec[ EndIndex ].m_type == Node::WALL )
		{
			return false;
		}

	}

	//------------------------------------------------

	int CurrentSquare = -1; // This changes as we go along.
	m_OpenList.Add( StartIndex );
	TrueNodeVec[ StartIndex ].m_G = 0.f;
	TrueNodeVec[ StartIndex ].m_H = CalculateH(StartIndex, EndIndex);
	TrueNodeVec[ StartIndex ].m_F = TrueNodeVec[ StartIndex ].m_G + TrueNodeVec[ StartIndex ].m_H; //@doom
	TrueNodeVec[ StartIndex ].m_Parent= -1;

	std::vector<int> Neighbors;
	assert(m_OpenList.Size() == 1); // always starts w/ one.
	assert( GetCurrentSquare() == StartIndex ); // it has to be [first time thru]
	while ( m_OpenList.Size() > 0 )
	{
		CurrentSquare = GetCurrentSquare();
		// Swap CurrentSquare to the closed list.
		RemoveFromOpenList( CurrentSquare );
		AddToClosedList( CurrentSquare );
		if ( CurrentSquare == EndIndex)
		{
			// Path Solution Found!
			SolutionFound = true;
			break;
		}
		// Gather adjacent squares to the current square.
		FindNeighbors( CurrentSquare, useEightWay, true, Neighbors );
		for (int k = 0; k < (int)Neighbors.size(); ++k)
		{
			// For each neighbor, if it is a wall or is on the closed list, ignore it.
			//  [updated for ignoreWalls thing]

			if ( IsOnClosedList( Neighbors[k] ) )
			{
				// Do Nothing
				continue;
			}

			if ( !ignoreWalls && TrueNodeVec[ Neighbors[k] ].m_type == Node::WALL ) // if we are NOT ignoring walls and this IS a wall.
			{
				// Do Nothing
				continue;
			}

			// Now see if it is on the open list already or not.
			if ( IsOnOpenList( Neighbors[k] ) )
			{
				// Is Already On Open List
				// See if this path is better, use G as the decider.
				// Lower G means this is a better path.
				// If G is lower, change parent of square to the current square,
				// and update the G and F scores of the square.
				assert( CalculateG( StartIndex, Neighbors[k] ) == TrueNodeVec[ Neighbors[k] ].m_G ); // if this assert never fires, this proves my theory as to why a better G is never found.  Needs to try a new parent/path to happen!!! [theory proven]
				// must TRY it as if this were the path hence a new parent]  
				// Temporarily give it a new parent.
				int currentParentOfNK = TrueNodeVec[ Neighbors[k] ].m_Parent; // save me.
				assert( IsIndexValid(currentParentOfNK) ); // start was long ago put on closed list, so the -1 parent should not arise anyway.
				TrueNodeVec[ Neighbors[k] ].m_Parent = CurrentSquare; // Pretend / What if I were your parent / hence a new/diff path?
				float localG = CalculateG( StartIndex, Neighbors[k] );

				if (localG < TrueNodeVec[ Neighbors[k] ].m_G)
				{
					// [already is thanks to above pretend, keep it this way]: TrueNodeVec[ Neighbors[k] ].m_Parent = CurrentSquare;
					TrueNodeVec[ Neighbors[k] ].m_G = localG;
					TrueNodeVec[ Neighbors[k] ].m_F = TrueNodeVec[ Neighbors[k] ].m_G + TrueNodeVec[ Neighbors[k] ].m_H;
				}
				else
				{
					// Wasn't any better, restore to former glory.
					TrueNodeVec[ Neighbors[k] ].m_Parent = currentParentOfNK;
				}
			}
			else
			{
				// Is not already on Open List.
				// Add to Open List.
				// CurrentSquare is its parent.
				// Calculate F, G, and H [done in FN Call]
				AddToOpenList( Neighbors[k], CurrentSquare, StartIndex, EndIndex );
			}
		}

	} // end of mega while.



	if (SolutionFound)
	{
		// Realize that we flip/flopped Start and End at top so now it is okay!
		int it = EndIndex;
		while (it != StartIndex)
		{
			solVec.push_back(it);
			it = TrueNodeVec[it].m_Parent;
		}
		solVec.push_back(StartIndex);
	}
	else
	{
		// No Solution!!
		assert(m_OpenList.Size() == 0);
		assert(! IsOnOpenList(EndIndex) );
		assert(! IsOnClosedList(EndIndex) );
	}

	return SolutionFound;
}

int Astar::GetStartIndex()
{
	assert( IsIndexValid(m_Start) );
	return m_Start;
}

int Astar::GetEndIndex()
{
	assert( IsIndexValid(m_End) );
	return m_End;
}

int Astar::GetCurrentSquare()
{
	// Returns index to square with lowest F cost.
	assert( m_OpenList.Size() > 0 );
	// An element of 'm_OpenList' is an INDEX into the NodeVec.

	// Okay, time to pay the piper.  Find the member of the open list
	// with the smallest f cost.  This is a linear search.
	// This could be speed up if we did had some cache, etc... BUT,
	// compared to the savings due to my map in other places, this is not bad.
	// Note to self: In a future update, consider using a cache, or perhaps a way to sort
	// by f-cost.  Decide if the memory overhead and/or time spent to sort is REALLY
	// worth it or not.
	std::map<int,bool>::iterator it = m_OpenList.GetPtrToMapItself()->begin();

	float minF = TrueNodeVec[ it->first ].m_F;
	int minI = it->first;
	++it;

	while (it != m_OpenList.GetPtrToMapItself()->end())
	{
		assert( IsIndexValid( it->first ) );
		if (TrueNodeVec[ it->first ].m_F < minF)
		{
			minF = TrueNodeVec[ it->first ].m_F;
			minI = it->first;
		}		
		++it;
	}

	assert( IsIndexValid(minI) );
	return minI; // an index directly to the NodeVec.
}

void Astar::RemoveFromOpenList(int i)
{
	assert(IsOnOpenList(i));
	int size = (int)m_OpenList.Size();
	m_OpenList.Remove(i);
	assert( (int)m_OpenList.Size() == size - 1);
}

void Astar::RemoveFromClosedList(int i)
{
	assert(IsOnClosedList(i));
	int size = (int)m_ClosedList.Size();
	m_ClosedList.Remove(i);
	assert( (int)m_ClosedList.Size() == size - 1);
}

void Astar::AddToOpenList(int i, int Parent, int StartIndex, int EndIndex)
{
	assert(!IsOnOpenList(i));
	assert( IsIndexValid(i) );
	assert( IsIndexValid(Parent) );
	assert( IsIndexValid(StartIndex) );
	assert( IsIndexValid(EndIndex) );
	m_OpenList.Add(i);
	TrueNodeVec[i].m_Parent = Parent;
	// calc F now
	// F is based on G & H.
	TrueNodeVec[i].m_H = CalculateH(i, EndIndex);
	TrueNodeVec[i].m_G = CalculateG(StartIndex, i);
	TrueNodeVec[i].m_F = TrueNodeVec[i].m_G + TrueNodeVec[i].m_H;
}

void Astar::AddToClosedList(int i)
{
	assert(!m_ClosedList.Find(i));
	assert( IsIndexValid(i) );
	m_ClosedList.Add(i);
}

void Astar::DuplicateList( const std::vector<int> &Original, std::vector<int> &CopyIntVecRet )
{
	CopyIntVecRet.clear();
	for (int i = 0; i < (int)Original.size(); ++i)
	{
		CopyIntVecRet.push_back( Original[i] );
	}
	assert( CopyIntVecRet.size() == Original.size() );
}

bool Astar::IsOnOpenList(int i)
{
	assert( IsIndexValid(i) );
	return m_OpenList.Find(i);
}

bool Astar::IsOnClosedList(int i)
{
	return m_ClosedList.Find(i);
}

int Astar::Convert_xy_to_i(int x, int y)
{
	// Takes x and y, returns an i
	assert(x >= 0 && y >= 0);
	assert( IsIndexValid( (y * m_width) + x ) );
	return (y * m_width) + x;
}

void Astar::Convert_i_to_xy(int i, int &xRet, int &yRet)
{
	// Takes i, returns an x and y.
	assert( IsIndexValid(i) );
	xRet = i;
	yRet = 0;
	// take away UNTIL is <= w - 1, not WHILE
	while (xRet > m_width - 1)
	{
		xRet -= m_width;
		++yRet;
	}
	assert( i == Convert_xy_to_i(xRet, yRet) );
	assert( xRet >= 0 ); // 10-05-07
	assert( yRet >= 0 ); // 10-05-07
}

bool Astar::IsWall(const int i)
{
	assert( IsIndexValid(i) );
	return (TrueNodeVec[i].m_type == Node::WALL);
}

float Astar::EuclideanH(int i, int j)
{
	assert( IsIndexValid(i) );
	assert( IsIndexValid(j) );
	int iX; int iY; int jX; int jY;
	Convert_i_to_xy(i, iX, iY);
	Convert_i_to_xy(j, jX, jY);
	// d = sqrt of [   (x2-x1)^2  + (y2 - y1)^2             ]
	float mag =  (float)(    ( (jX - iX) * (jX - iX) ) + ( (jY - iY) * (jY - iY) )    );
	mag = sqrt(mag);
	return mag;
}


void Astar::ManhattanH(int i, int j, int &absDeltaX, int &absDeltaY)
{
	// Manhattan is not a good idea to use most of the time, but I left
	// it here incase I ever feel like using it for some reason.  It is faster,
	// but less accurate of a heuristic.
	assert( IsIndexValid(i) );
	assert( IsIndexValid(j) );
	int iX; int iY; int jX; int jY;
	Convert_i_to_xy(i, iX, iY);
	Convert_i_to_xy(j, jX, jY);
	absDeltaX = abs( jX - iX );
	absDeltaY = abs( jY - iY );
	assert(absDeltaX >= 0);
	assert(absDeltaY >= 0);
}


float Astar::CalculateH(int i, int j)
{
	// i is where you are, j is end.
	assert( IsIndexValid(i) );
	assert( IsIndexValid(j) );
	assert(  EuclideanH(i,j) == EuclideanH(j,i)  );
	return EuclideanH(i,j);
	///int absDeltaX; int absDeltaY;
	///ManhattanH(i, j, absDeltaX, absDeltaY);
	///return (absDeltaX + absDeltaY) * 10;
}

float Astar::CalculateG(int i, int j)
{
	// i is start, j is where you are.
	assert( IsIndexValid(i) );
	assert( IsIndexValid(j) );
	float sum = 0.f;

	while (j != i) // while not at the beginning yet.
	{
		sum += CalculateMovementCost( TrueNodeVec[j].m_Parent, j );
		j = TrueNodeVec[j].m_Parent;
	}
	return sum;
}

float Astar::CalculateMovementCost(int i, int j)
{
	// Cost to move from i to j.
	assert( IsIndexValid(i) );
	assert( IsIndexValid(j) );
	// Find out if we went up or down or sideways, etc... to go from i to j.

	if (DoesExistUp(i) && OffsetUp(i) == j)
		return 1.f;

	if (DoesExistLeft(i) && OffsetLeft(i) == j)
		return 1.f;

	if (DoesExistDown(i) && OffsetDown(i) == j)
		return 1.f;

	if (DoesExistRight(i) && OffsetRight(i) == j)
		return 1.f;

	return 1.414f;
}

bool Astar::IsInSolVec(const int findIndex, const int startSearchIndex, std::vector<int> solVec)
{
	// findIndex is an index into the NODEvec.  startSearchIndex is index into the SOLvec.
	// Given a solvec, just goes thru it and sees if this index exists.  Starts at startIndex.
	for (int i = startSearchIndex; i < (int)solVec.size(); ++i)
	{
		if (solVec[i] == findIndex)
			return true;
	}
	return false;
}

void Astar::SecureBorders(void)
{
	const int nodes = m_width * m_height;
	for (int i = 0; i < nodes; ++i)
	{
		if ( (!DoesExistUp(i)) || (!DoesExistRight(i))  || (!DoesExistDown(i)) || (!DoesExistLeft(i))  )
		{
			TrueNodeVec[i].m_type = Node::WALL;
		}
	}
}

std::vector<Node> *Astar::GetTrueNodeVecPtr(void)
{
	if (m_externalNodeVecPtr != NULL)
	{
		assert( m_NodeVec.size() == 0 ); // we want it this way.  Otherwise, infinite recursion.
		return m_externalNodeVecPtr;
	}
	else
	{
		//assert( m_NodeVec.size() > 0 ); [may not exist yet, may just have been cleared, etc...]
		return &m_NodeVec; // We want it this way.
	}
}

void Astar::LoadMapFromFile(std::string file, char empty, char start, char end, char wall)
{
	TrueNodeVec.clear();
	int initialCount = -1;
	int h = 0;
	std::string entireLine;

	// Open in reading mode
	std::ifstream fin( file.c_str() );
	// Grab an entire line save as a string

	while (! fin.eof() )
	{
		getline(fin, entireLine);

		if (initialCount != -1 && (int)entireLine.size() != initialCount)
		{
			break;
		}

		if (initialCount == -1)
		{
			initialCount = (int)entireLine.size();
			m_width = initialCount;
		}

		++h;

		for (int z = 0; z < (int)entireLine.size(); ++z)
		{
			if ( entireLine[z] == empty)
			{
				TrueNodeVec.push_back( Node() );
				TrueNodeVec.back().m_type = Node::EMPTY;
			}
			else if ( entireLine[z] == start)
			{
				TrueNodeVec.push_back( Node() );
				TrueNodeVec.back().m_type = Node::START;
			}
			else if ( entireLine[z] == end)
			{
				TrueNodeVec.push_back( Node() );
				TrueNodeVec.back().m_type = Node::END;
			}
			else if ( entireLine[z] == wall)
			{
				TrueNodeVec.push_back( Node() );
				TrueNodeVec.back().m_type = Node::WALL;
			}
		}

		entireLine.clear();
	}

	fin.close();
	m_height = h;
	assert(TrueNodeVec.size() == m_height * m_width);

	// Now I need to go and turn on m_End and m_Start [the real indicators now]
	const int nodes = m_width * m_height;
	for (int i = 0; i < nodes; ++i)
	{
		if (TrueNodeVec[i].m_type == Node::START)
			m_Start = i;
		else if (TrueNodeVec[i].m_type == Node::END)
			m_End = i;
	}
	// Realize that a loaded in map SHOULD'NT have a start and end in it anyway for pratical purposes.
	// Plus, the S and E must be 2 different squares in the map.

}


void Astar::DebugSendNodesToFile(std::string file)
{
	std::ofstream fout( file.c_str() );
	assert( (int)TrueNodeVec.size() == m_width * m_height );
	fout << "Width: " << m_width << std::endl;
	fout << "Height: " << m_height << std::endl;
	fout << std::endl;
	for (int i = 0; i < (int)TrueNodeVec.size(); ++i)
	{
		fout << "Node # " << i << std::endl;
		fout << "Node Type: " << TrueNodeVec[i].m_type << std::endl;
		fout << "F: "<< TrueNodeVec[i].m_F << std::endl;
		fout << "G: "<< TrueNodeVec[i].m_G << std::endl;
		fout << "H: "<< TrueNodeVec[i].m_H << std::endl;
		fout << "Node's Parent: " << TrueNodeVec[i].m_Parent << std::endl;
		fout << std::endl;
		fout << std::endl;
	}
	fout.close();

}





